/*
 *	This abreviated version of ajax.js only supports text mode
 *	communication.
 *
 * File: miniajax.js
 *
 *	AJAX  Asynchronous JavaScript and XML
 *
 *	An interface that minimally supports the AJAX functionality of the
 *	XMLHttpRequest object. Typical usage is:
 *
 * 		> var miniajax = new Miniajax();
 * 		> miniajax.setAsync(false);
 * 		> miniajax.addParameter("parameter", "value");
 * 		> miniajax.callURL("upload.php", completionFunction);
 *
 * Copyright:
 *
 * 	Copyright (c) 2010 by Timothy Wagner. All Rights Reserved.
 *
 */

/*
 * Class: Miniajax Class
 *
 * 	AJAX  Asynchronous JavaScript and XML
 *
 * Constructor: Miniajax
 *
 * 	Create instance of the Miniajax class
 *
 */
function Miniajax()
	{
	var that	= this;
	var xmlHttp	= null;
	var onComplete	= null;
	var async	= true;
	var url		= "";
	var params	= {};

	// Correct if called as function rather than with 'new'
	if (!(this instanceof Miniajax)) 
		return new Miniajax();

	/*
	 * Cross browser Ajax requests from:
	 * http://msdn.microsoft.com/en-us/library/ms537505(VS.85).aspx
	 */

	// If IE7, Mozilla, Safari, and so on: Use native object.
	if (window.XMLHttpRequest)
		xmlHttp = new XMLHttpRequest();

	// ...otherwise, use the ActiveX control for IE5.x and IE6.
	else if (window.ActiveXObject)
		xmlHttp = new ActiveXObject( 'MSXML2.XMLHTTP.3.0');

	if (!xmlHttp)
		{
		alert ("Browser does not support HTTP Request");
		return;
		}

	/*
	 * Method: addParameter
	 *
	 *	Add variable/value pairs to the ajax query
	 *
	 * Parameters:
	 *
	 * 	name	- name of a parameter
	 * 	value	- value of a parameter
	 *
	 */
	this.addParameter = function(name, value)
		{
		params[encodeURIComponent(name)] = encodeURIComponent(value);
		}

	/*
	 * Method: setAsync
	 *
	 *	Set asynchronous transfer mode
	 *
	 *	Determine whether the ajax request will be run
	 *	synchronously or asynchronously.
	 *
	 * Parameter:
	 *
	 * 	isAsync		- [ true | false ] true if ajax call is
	 * 			  to be executed asynchronously.
	 *
	 */
	this.setAsync = function(isAsync)
		{
		async = isAsync;
		}

	/*
	 * Method: callURL
	 *
	 *	Send the AJAX request
	 *
	 * Parameters:
	 *
	 * 	url		- name of remote file
	 * 	onCompleteFunc	- callback function executed on completion
	 *
	 */
	this.callURL = function(url, onCompleteFunc)
		{
		that.url = url;

		// Creates unique URL to prevent caching. This may
		// be redundant since the POST method is being used.
		url += "?sid=" + Math.random();

		// Call-back functions to process AJAX changes 
		if (async)
			xmlHttp.onreadystatechange = stateChanged;
		onComplete = onCompleteFunc;

		xmlHttp.open("POST", url, async);
		var body = "";

		for (var name in params)
			{
			if (body.length != 0)
				body += "&";
			body += name + "=" + params[name];
			}

		xmlHttp.setRequestHeader("Content-type",
			"application/x-www-form-urlencoded");
		xmlHttp.setRequestHeader("Content-length", body.length);
		xmlHttp.setRequestHeader("Connection", "close");

		// Send the request.
		try	{
			xmlHttp.send(body);
			}
		catch(e)
			{
			// Firefox fires this error on location change if
			// ajax is in process.
			// 	"Component_returned_failure_code:_0x80040111_
			// 	(NS_ERROR_NOT_AVAILABLE)"
			// See page for more info:
			//	 http://helpful.knobs-dials.com/index.php/
			//		0x80004005_%28NS_ERROR_FAILURE%29
			//		_and_other_firefox_errors
			// Ignore this Firefox error:
			if (e.name == "NS_ERROR_FAILURE")
				return;
			// Report all others:
			alert("Error name: " + e.name +
				". Error message: " + e.message);
			}

		// If the request was executed synchronously, then we must
		// process the return results directly rather than waiting
		// for the onreadystatechange event (which will never come).
		if (!async)
			stateChanged();
		}

	function stateChanged() 
		{ 
		if (xmlHttp.readyState == 4 || xmlHttp.readyState == "complete")
			{
			if (xmlHttp.status == 200)
				onComplete(xmlHttp.responseText);

			// In an asynchronous request where the user navigates
			// to a new page before the request completes, the
			// status will be zero. This can be safely ignored.
			// All other status values are reported as an error.
			else if (xmlHttp.status !== 0)
				{
				alert('Error: ' + xmlHttp.status +
					"\nLoading File: " + that.url);
				}
			}
		}
	}
